/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * video window handling functions
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "common.h"
#include "xine-toolkit/xitk_x11.h"
#include "xine-toolkit/window.h"
#include "xine-toolkit/_backend.h"
#include "kbindings.h"
#include "videowin.h"
#include "tvout.h"
#include "actions.h"
#include "event.h"
#include "errors.h"
#include "osd.h"

#include "oxine/oxine_event.h"
#include "oxine/oxine.h"

#define EST_KEEP_VALID  10	  /* #frames to allow for changing fps */
#define EST_MAX_JITTER  0.01	  /* maximum jitter to detect valid fps */
#define EST_MAX_DIFF    0.01      /* maximum diff to detect valid fps */
#define ABS(x) ((x)>0?(x):-(x))

/** private *****************************************************************/

struct xui_vwin_st {
  gGui_t                *gui;
  xitk_t                *xitk;

  char                   window_title[1024];
  int                    current_cursor;  /* arrow or hand */
  int                    cursor_visible;

  int                    separate_display; /* gui and video window use different displays */
  int                    wid; /* use this window */
  xitk_be_display_t     *gui_be_display, *video_be_display;
  Display               *video_display;

  struct {
    xitk_window_t       *xwin;
    xitk_be_window_t    *bewin;
    uintptr_t            window;
  }                      win, oldwin;

  char                  *prefered_visual;
  Visual	        *visual;          /* Visual for video window               */
  struct {
    const char *normal, *fullscreen, *borderless;
  } res_name;

  int                    video_width;     /* size of currently displayed video     */
  int                    video_height;

  int                    frame_width;     /* frame size, from xine-lib */
  int                    frame_height;

#if 0
  double                 video_duration;  /* frame duratrion in seconds */
  double                 video_average;   /* average frame duration in seconds */
  double                 use_duration;    /* duration used for tv mode selection */
  int                    video_duration_valid; /* is use_duration trustable? */
#endif
  double                 pixel_aspect;
  xitk_rect_t            win_r;           /* current location, size of non-fullscreen window */
  xitk_rect_t            old_win_r;
  int                    output_width;    /* output video window width/height      */
  int                    output_height;
  int                    output_changed;

  int                    stream_resize_window; /* Boolean, 1 if new stream resize output window */
  int                    zoom_small_stream; /* Boolean, 1 to double size small streams */

  /* current mode */
  uint32_t               mode;
  /* what we want next */
  uint32_t               new_mode;
  /* beware of complex logic:
   * 1. start with all 0.
   * 2. clear the bits we want to change.
   * 3. set the bits the server has applied ok.
   * 4. detect back changes by user/window manager. */
  uint32_t               watch_mode;

  int                    fullscreen_width;
  int                    fullscreen_height;

  int                    visible_width;   /* Size of currently visible portion of screen */
  int                    visible_height;  /* May differ from fullscreen_* e.g. for TV mode */
  double                 visible_aspect;  /* Pixel ratio of currently visible screen */

#ifdef HAVE_XINERAMA
  const char            *xinerama_pref;
  xitk_rect_t            xinerama_fullscreen;
  uint32_t               xinerama_flags;
  uint8_t                xinerama_helipad[4];
#endif

  int                    desktopWidth;    /* desktop width */
  int                    desktopHeight;   /* desktop height */
  int                    depth;
  int                    show;

  xitk_register_key_t    widget_key;

#ifdef HAVE_XF86VIDMODE
  int                    XF86VidMode;
#endif

  int                    hide_on_start; /* user use '-H' arg, don't map
					   video window the first time */

  pthread_t              second_display_thread;
  int                    second_display_running;

  pthread_mutex_t        mutex;

  union {
#if defined(HAVE_X11)
    x11_visual_t          x11;
#endif
#if defined(HAVE_WAYLAND) && defined(XINE_VISUAL_TYPE_WAYLAND)
    xine_wayland_visual_t wl;
#endif
  } xine_visual;

  int                     oxine_res;
};

static unsigned int vwin_mag_mul (unsigned int a, unsigned int b) {
#if 1
  return ((uint64_t)a * b + VWIN_MAG_1 / 2) >> VWIN_MAG_LD;
#else
  /* works as well :-) */
  unsigned int a1 = a >> VWIN_MAG_LD, a0 = a & (VWIN_MAG_1 - 1), b1 = b >> VWIN_MAG_LD, b0 = b & (VWIN_MAG_1 - 1);
  return ((a1 * b1) << VWIN_MAG_LD) + (a1 * b0) + (a0 * b1) + ((a0 * b0) >> VWIN_MAG_LD);
#endif
}

static unsigned int vwin_mag_get (unsigned int num, unsigned int den) {
  return den ? (((num & (VWIN_MAG_1 - 1)) << VWIN_MAG_LD) + (den >> 1)) / den
               + ((((num >> VWIN_MAG_LD) + (den >> 1)) / den) << VWIN_MAG_LD)
             : VWIN_MAG_1;
}

/** Raise above the Gnome toolbar and similar stuff.
 *  level 1: all windows except non fullscreen video.
 *  levels 2 and 3: all windows. this also prevents pulling up other applications,
 *  so its probably better to use window manager settings there.
 *  at least kwin seems to work fine without all this :-) */
static void _vwin_layer_above (xui_vwin_t *vwin) {
  xitk_tagitem_t tags[] = {
    {XITK_TAG_LAYER_ABOVE, 0},
    {XITK_TAG_END, 0}
  };
  switch (vwin->gui->layer_above & 3) {
    default:
      break;
    case 1:
      if (vwin->mode & WINDOWED_MODE)
        break;
      {
        uint32_t type = vwin->video_be_display->wm_type;
        if (!(type & WM_TYPE_EWMH_COMP))
          break;
      }
      /* fall through */
    case 2:
    case 3:
      tags[0].value = 1;
      break;
  }
  vwin->win.bewin->set_props (vwin->win.bewin, tags);
}

static void _vwin_set_input_focus (xui_vwin_t *vwin) {
  if (xitk_window_flags (vwin->win.xwin, 0, 0) & XITK_WINF_VISIBLE)
    xitk_window_set_input_focus (vwin->win.xwin);
}

static int _vwin_is_visible (xui_vwin_t *vwin) {
  static const uint8_t map[6] = {0, 1, 0, 2, 2, 2};
  int visible = xitk_window_flags (vwin->win.xwin, 0, 0) & XITK_WINF_VISIBLE;
  /* user may have changed this via task bar. */
  vwin->show = map[vwin->show + (visible ? 3 : 0)];
  return vwin->show;
}

static void _vwin_resize_cb (void *data, xine_cfg_entry_t *cfg) {
  xui_vwin_t *vwin = data;
  vwin->stream_resize_window = cfg->num_value;
}

static void _vwin_zoom_small_cb (void *data, xine_cfg_entry_t *cfg) {
  xui_vwin_t *vwin = data;
  vwin->zoom_small_stream = cfg->num_value;
}

#ifdef HAVE_XF86VIDMODE
static void _adjust_modeline(xui_vwin_t *vwin) {

  if (vwin->new_mode & WINDOWED_MODE) {
    xitk_change_video_mode(vwin->xitk, vwin->win.xwin, -1, -1);
  } else if ((!(vwin->new_mode & WINDOWED_MODE)) || (!(vwin->mode & WINDOWED_MODE))) {
    xitk_change_video_mode(vwin->xitk, vwin->win.xwin, vwin->video_width, vwin->video_height);
  }

  vwin->fullscreen_width  = vwin->video_be_display->default_screen.r.width;
  vwin->fullscreen_height = vwin->video_be_display->default_screen.r.height;
  vwin->pixel_aspect = vwin->video_be_display->default_screen.ratio;

#ifdef DEBUG
  printf ("pixel_aspect: %f\n", vwin->pixel_aspect);
#endif
}
#endif /* HAVE_XF86VIDMODE */

/*
 * check if screen_number is in the list
 */
#ifdef HAVE_XINERAMA
static void _xinerama_parse_user_pref (xui_vwin_t *vwin, const char *pref) {
  const uint8_t *p = (const uint8_t *)pref;
  xitk_rect_t hull;
  const xitk_rect_t *list = vwin->video_be_display->screens;
  int num_list = vwin->video_be_display->num_screens;
  int want_first = -1, want_num = 0, have_num = 0;

  if (!list || (num_list <= 0)) {

    hull.x = hull.y = 0;
    hull.width = vwin->fullscreen_width;
    hull.height = vwin->fullscreen_height;

  } else {

    /* strange rule:
     * 1. ignore all screens lower than the first user one.
     * 2. skip all nonexisting user ones.
     * 3. build the hull of what is left.
     * 4. if nothing left, fall back to first existing. */

    while (*p && ((*p ^ '0') >= 10u))
      p++;
    if (*p) {
      uint32_t v = 0;
      uint8_t z;
      while ((z = *p ^ '0') < 10u)
        v = v * 10u + z, p++;
      want_first = v;
      want_num = 1;
    }
    if (XITK_0_TO_MAX_MINUS_1 (want_first, num_list)) {
      have_num = 1;
      hull = list[want_first];
      hull.width += hull.x;
      hull.height += hull.y;
      while (*p) {
        uint32_t v;
        uint8_t z;
        while (*p && ((*p ^ '0') >= 10u))
          p++;
        if (!*p)
          break;
        want_num++;
        v = 0;
        while ((z = *p ^ '0') < 10u)
          v = v * 10u + z, p++;
        if (XITK_0_TO_MAX_MINUS_1 ((int)v - want_first, num_list - want_first)) {
          have_num++;
          if (hull.x > list[v].x)
            hull.x = list[v].x;
          if (hull.y > list[v].y)
            hull.y = list[v].y;
          if (hull.width < list[v].x + list[v].width)
            hull.width = list[v].x + list[v].width;
          if (hull.height < list[v].y + list[v].height)
            hull.height = list[v].y + list[v].height;
        }
      }
      hull.width -= hull.x;
      hull.height -= hull.y;
    } else {
      hull = list[0];
      want_first = -1;
    }

  }

  pthread_mutex_lock (&vwin->mutex);
  vwin->xinerama_pref = pref;
  vwin->xinerama_flags &= ~16;
  vwin->xinerama_flags |= (num_list > 0) ? 16 : 0;
  if (!(vwin->xinerama_flags & 1))
    vwin->xinerama_fullscreen.x = hull.x;
  if (!(vwin->xinerama_flags & 2))
    vwin->xinerama_fullscreen.y = hull.y;
  if (!(vwin->xinerama_flags & 4))
    vwin->xinerama_fullscreen.width = hull.width;
  if (!(vwin->xinerama_flags & 8))
    vwin->xinerama_fullscreen.height = hull.height;
  pthread_mutex_unlock (&vwin->mutex);

  if (vwin->gui->verbosity >= 2)
    printf ("video_window.xinerama.init.%s (want=%d, have=%d, x=%d, y=%d, w=%d, h=%d).\n",
      (want_first >= 0) ? "user" : "fallback", want_num, have_num,
      hull.x, hull.y, hull.width, hull.height);
}

static void _xinerama_new_user_pref (void *data, xine_cfg_entry_t *e) {
  xui_vwin_t *vwin = (xui_vwin_t *)data;

  _xinerama_parse_user_pref (vwin, e->str_value);
}

static void _xinerama_new_user_val (void *data, xine_cfg_entry_t *e) {
  uint8_t *land = (uint8_t *)data;
  int i = *land, *val;
  uint32_t f = 1 << i;
  xui_vwin_t *vwin;
  const char *pref;

  land -= i;
  xitk_container (vwin, land, xinerama_helipad[0]);
  val = &vwin->xinerama_fullscreen.x;
  pthread_mutex_lock (&vwin->mutex);
  pref = vwin->xinerama_pref;
  if (e->num_value > -8192) {
    val[i] = e->num_value;
    vwin->xinerama_flags |= f;
  } else {
    vwin->xinerama_flags &= ~f;
  }
  pthread_mutex_unlock (&vwin->mutex);
  _xinerama_parse_user_pref (vwin, pref);
}

static void _init_xinerama (xui_vwin_t *vwin) {
  /* Spark
   * some Xinerama stuff
   * I want to figure out what fullscreen means for this setup
   */

  vwin->xinerama_flags = 0;
  memcpy (vwin->xinerama_helipad, "\x00\x01\x02\x03", 4);
  if (!vwin->video_be_display->screens)
    return;

  /* Xinerama Detected */

  if (vwin->video_be_display->num_screens > 0) {
    int v;

    v = xine_config_register_num (vwin->gui->xine, "gui.xinerama_fullscreen_x", -8192,
        _("x coordinate for xinerama fullscreen (-8192 = autodetect)"),
        CONFIG_NO_HELP, CONFIG_LEVEL_EXP, _xinerama_new_user_val, vwin->xinerama_helipad + 0);
    if (v > -8192)
        vwin->xinerama_flags |= 1, vwin->xinerama_fullscreen.x = v;

    v = xine_config_register_num (vwin->gui->xine, "gui.xinerama_fullscreen_y", -8192,
        _("y coordinate for xinerama fullscreen (-8192 = autodetect)"),
        CONFIG_NO_HELP, CONFIG_LEVEL_EXP, _xinerama_new_user_val, vwin->xinerama_helipad + 1);
    if (v > -8192)
        vwin->xinerama_flags |= 2, vwin->xinerama_fullscreen.y = v;

    v = xine_config_register_num (vwin->gui->xine, "gui.xinerama_fullscreen_width", -8192,
        _("width for xinerama fullscreen (-8192 = autodetect)"),
        CONFIG_NO_HELP, CONFIG_LEVEL_EXP, _xinerama_new_user_val, vwin->xinerama_helipad + 2);
    if (v > -8192)
        vwin->xinerama_flags |= 4, vwin->xinerama_fullscreen.width = v;

    v = xine_config_register_num (vwin->gui->xine, "gui.xinerama_fullscreen_height", -8192,
        _("height for xinerama fullscreen (-8192 = autodetect)"),
        CONFIG_NO_HELP, CONFIG_LEVEL_EXP, _xinerama_new_user_val, vwin->xinerama_helipad + 3);
    if (v > -8192)
        vwin->xinerama_flags |= 8, vwin->xinerama_fullscreen.height = v;

    _xinerama_parse_user_pref (vwin, xine_config_register_string (vwin->gui->xine,
       "gui.xinerama_use_screens", "0 1",
      _("Screens to use in order to do a very fullscreen in xinerama mode. (example 0 2 3)"),
      _("Example, if you want the display to expand on screen 0, 2 and 3, enter 0 2 3"),
      CONFIG_LEVEL_EXP, _xinerama_new_user_pref, vwin));

  } else {
    /* no Xinerama */
    if (vwin->gui->verbosity)
      printf ("Display is not using Xinerama.\n");
    vwin->xinerama_fullscreen.x      = 0;
    vwin->xinerama_fullscreen.y      = 0;
    vwin->xinerama_fullscreen.width  = vwin->fullscreen_width;
    vwin->xinerama_fullscreen.height = vwin->fullscreen_height;
  }
}

static void _detect_xinerama_pos_size (xui_vwin_t *vwin, xitk_rect_t *r) {
  if (vwin->video_be_display->screens) {

    if (vwin->new_mode & FULLSCR_XI_MODE) {

      r->x = vwin->xinerama_fullscreen.x;
      r->y = vwin->xinerama_fullscreen.y;
      vwin->fullscreen_width = r->width  = vwin->xinerama_fullscreen.width;
      vwin->fullscreen_height = r->height = vwin->xinerama_fullscreen.height;
      if (vwin->gui->verbosity >= 2)
        printf ("video_window.xinerama.fullscreen (%d, %d, %d, %d).\n",
          r->x, r->y, r->width, r->height);

    } else {

      /* stay on the screen we (or the ouse) were on. */
      int n;
      const char *name;
      *r = vwin->old_win_r;
      n = vwin->video_be_display->find_screen (vwin->video_be_display, r);
      if (vwin->new_mode & WINDOWED_MODE) {
        int d;
        /* make sure middle is inside, XITK_WINF_FENCED_IN will do the rest. */
        d = r->x - (vwin->video_width >> 1);
        if (r->x > d + r->width - 1)
          r->x = d + r->width - 1;
        else if (r->x < d)
          r->x = d;
        d = r->y - (vwin->video_height >> 1);
        if (r->y > d + r->height - 1)
          r->y = d + r->height - 1;
        else if (r->y < d)
          r->y = d;
        r->width  = vwin->video_width;
        r->height = vwin->video_height;
        name = "windowed";
      } else {
        vwin->fullscreen_width = r->width;
        vwin->fullscreen_height = r->height;
        name = "full";
      }
      if (vwin->gui->verbosity >= 2)
        printf ("video_window.xinerama.use_screen.%s (#%d, %d, %d, %d, %d).\n",
          name, n, r->x, r->y, r->width, r->height);

    }
  } else { /* !vwin->xinerama */
    r->x = r->y = 0;
    if (vwin->new_mode & WINDOWED_MODE) {
      r->width  = vwin->win_r.width;
      r->height = vwin->win_r.height;
    } else {
      r->width  = vwin->fullscreen_width;
      r->height = vwin->fullscreen_height;
    }
  }
}
#endif /* HAVE_XINERAMA */

/*
 * hide/show cursor in video window
 */
static void _vwin_cursor_on (xui_vwin_t *vwin) {
  if (vwin->gui->use_root_window)
    return;
  vwin->gui->cursor_visible = 5; /** << show 5 seconds after last move/click. XXX: make configurable? */
  vwin->cursor_visible = 1;
  xitk_window_define_window_cursor (vwin->win.xwin,
    (vwin->current_cursor == CURSOR_ARROW) ? xitk_cursor_default : xitk_cursor_hand2);
}

/*
 * Translate screen coordinates to video coordinates
 */
static int _vwin_translate_point (xui_vwin_t *vwin, int xy[2]) {
  x11_rectangle_t rect;
  float           xf,yf;
  float           scale, width_scale, height_scale,aspect;

  xitk_rect_t     wr = {0, 0, 0, 0};

  if (!vwin->gui->vo_port)
    return 0;

  rect.x = xy[0];
  rect.y = xy[1];
  rect.w = 0;
  rect.h = 0;

  if (xine_port_send_gui_data (vwin->gui->vo_port,
    XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, (void*)&rect) != -1) {
    /* driver implements vwin->gui->video coordinate space translation, use it */
    xy[0] = rect.x;
    xy[1] = rect.y;
    return 1;
  }

  /* Driver cannot convert vwin->gui->video space, fall back to old code... */

  pthread_mutex_lock (&vwin->mutex);
  xitk_window_get_window_position (vwin->win.xwin, &wr);
  if (wr.width < 1 || wr.height < 1) {
    pthread_mutex_unlock (&vwin->mutex);
    return 0;
  }

  /* Scale co-ordinate to image dimensions. */
  height_scale = (float)vwin->video_height / (float)wr.height;
  width_scale  = (float)vwin->video_width / (float)wr.width;
  aspect       = (float)vwin->video_width / (float)vwin->video_height;
  if (((float)wr.width / (float)wr.height) < aspect) {
    scale    = width_scale;
    xf       = (float)xy[0] * scale;
    yf       = (float)xy[1] * scale;
    /* wwin=wwin * scale; */
    wr.height = wr.height * scale;
    /* FIXME: The 1.25 should really come from the NAV packets. */
    xy[0] = xf * 1.25 / aspect;
    xy[1] = yf - ((wr.height - vwin->video_height) / 2);
    /* printf("wscale:a=%f, s=%f, x=%d, y=%d\n",aspect, scale,*video_x,*video_y);  */
  } else {
    scale    = height_scale;
    xf       = (float)xy[0] * scale;
    yf       = (float)xy[1] * scale;
    wr.width = wr.width * scale;
    /* FIXME: The 1.25 should really come from the NAV packets. */
    xy[0] = (xf - ((wr.width - vwin->video_width) /2)) * 1.25 / aspect;
    xy[1] = yf;
    /* printf("hscale:a=%f s=%f x=%d, y=%d\n",aspect,scale,*video_x,*video_y);  */
  }

  pthread_mutex_unlock (&vwin->mutex);

  return 1;
}

static void _vwin_adapt_size (xui_vwin_t *vwin, int really_new);

static int _vwin_oxine_event (xui_vwin_t *vwin, oxine_event_t *oe) {
  int r = oxine_event (vwin->gui->oxine, oe);

  if ((r & OXINE_EVENT_WIDGET_HELD) == vwin->oxine_res)
    return r & OXINE_EVENT_HANDLED;
  vwin->oxine_res = r & OXINE_EVENT_WIDGET_HELD;
  xitk_window_flags (vwin->win.xwin, XITK_WINF_FIXED_POS, xitk_bitmove (r, OXINE_EVENT_WIDGET_HELD, XITK_WINF_FIXED_POS));
  return r & OXINE_EVENT_HANDLED;
}

static int _vwin_handle_be_event (void *data, const xitk_be_event_t *e) {
  xui_vwin_t *vwin = data;
  oxine_event_t oe = {.data = NULL};

  if (vwin->gui->verbosity >= 3)
    printf ("video_window.event (type=%d, code=%u, x=%d, y=%d, w=%d, h=%d}.\n",
      e->type, (unsigned int)e->code, e->x, e->y, e->w, e->h);

  switch (e->type) {
    case XITK_EV_WIN_FLAGS:
      {
        uint32_t applied = xitk_bitmove (e->qual, XITK_WINF_FULLSCREEN, FULLSCR_MODE)
                         | xitk_bitmove (e->qual, XITK_WINF_DECORATED, BORDER_MODE);
        uint32_t match = (~applied ^ vwin->mode) & (FULLSCR_MODE | BORDER_MODE);
        vwin->watch_mode |= match;
        uint32_t user_change = ~match & vwin->watch_mode;
        if (user_change & FULLSCR_MODE) {
          /* user says "fullscreen" through the window manager.
           * this is _not_ the same as ours. even worse: trying to open menu thereafter
           * triggers another strange mode switch that does kill the menu immediately
           * twice, with some crashes due to an invalid widget list.
           * better switch again ourselves here. */
          if (vwin->gui->verbosity >= 2)
            printf ("video_window.fix_user_fullscreen (%s}.\n", (vwin->mode & FULLSCR_MODE) ? "off" : "on");
          pthread_mutex_lock (&vwin->mutex);
          vwin->watch_mode &= ~FULLSCR_MODE;
          vwin->new_mode = (vwin->mode & ~(WINDOWED_MODE | FULLSCR_XI_MODE | FULLSCR_MODE))
                         | ((vwin->mode & FULLSCR_MODE) ? WINDOWED_MODE : FULLSCR_MODE);
          _vwin_adapt_size (vwin, 1);
          _vwin_set_input_focus (vwin);
          osd_update (vwin->gui);
          /* Drawable has changed, update cursor visiblity */
          _vwin_cursor_on (vwin);
          pthread_mutex_unlock (&vwin->mutex);
        }
      }
      break;
    case XITK_EV_BUTTON_UP:
      oe.x = e->x;
      oe.y = e->y;
      oe.type = OXINE_EVENT_BUTTON;
      oe.key = OXINE_BUTTON1 | OXINE_KEY_RELEASE;
      if ((e->code == 1) && _vwin_oxine_event (vwin, &oe))
        return 1;
      if ((e->code == 1) && (e->qual & MODIFIER_DOUBLE_CLICK))
        gui_action_id (vwin->gui, ACTID_TOGGLE_FULLSCREEN);
      return 1;
    case XITK_EV_BUTTON_DOWN:
      {
        static const uint8_t b1[XITK_BUTTON_LASTCODE] = {
          [XITK_BUTTON_MIDDLE]      = ACTID_TOGGLE_VISIBLITY,
          [XITK_BUTTON_WHEEL_UP]    = ACTID_pVOLUME,
          [XITK_BUTTON_WHEEL_DOWN]  = ACTID_mVOLUME,
          [XITK_BUTTON_SIDE_LEFT]   = ACTID_WINDOWREDUCE,
          [XITK_BUTTON_SIDE_RIGHT]  = ACTID_WINDOWENLARGE
        };
        xine_event_t      event;

        if (vwin->gui->cursor_visible <= 0)
          _vwin_cursor_on (vwin);
        else
          vwin->gui->cursor_visible = 5;
        if (e->code == 3) {
          /* gui_handle_be_event () now does the menu. */
          if (vwin->separate_display) /* no menu here */
            return 1;
          break;
        } else if (e->code == 1) {
          xitk_gettime_tv (&event.tv);
        } else if (XITK_0_TO_MAX_MINUS_1 (e->code, XITK_BUTTON_LASTCODE) && b1[e->code]) {
          /* printf ("videowin: mouse wheel event %u.\n", e->more); */
          /* only ACTID_?VOLUME will use the num arg. */
          gui_action_args (vwin->gui, b1[e->code], ((e->qual & MODIFIER_SHIFT) ? 2 * e->more + 2 : e->more + 1), NULL);
          return 1;
        };

        oe.type = OXINE_EVENT_BUTTON;
	oe.key = OXINE_BUTTON1 | ((e->type == XITK_EV_BUTTON_UP) ? OXINE_KEY_RELEASE : 0);
        oe.x = e->x;
        oe.y = e->y;
        event.type = XINE_EVENT_INPUT_MOUSE_BUTTON;
        if (e->code != 1 || !_vwin_oxine_event (vwin, &oe)) {
          int p[2] = {e->x, e->y };
          if (_vwin_translate_point (vwin, p)) {
            xine_input_data_t input;
            input.x      = p[0];
            input.y      = p[1];
            input.button = e->code;
            event.stream      = vwin->gui->stream;
            event.data        = &input;
            event.data_length = sizeof (input);
            xine_event_send (vwin->gui->stream, &event);
          }
        }
      }
      return 1;
    case XITK_EV_ENTER:
    case XITK_EV_MOVE:
      {
        if (vwin->gui->cursor_visible <= 0)
          _vwin_cursor_on (vwin);
        else
          vwin->gui->cursor_visible = 5;

        oe.type = OXINE_EVENT_MOTION;
	oe.key = OXINE_BUTTON1 | ((e->qual & MODIFIER_BUTTON1) ? OXINE_KEY_RELEASE : 0);
        oe.x = e->x;
        oe.y = e->y;
        if (!_vwin_oxine_event (vwin, &oe)) {
          int p[2] = {e->x, e->y};
          if (_vwin_translate_point (vwin, p)) {
            xine_event_t event;
            xine_input_data_t input;
            input.x = p[0];
            input.y = p[1];
            input.button = 0; /*  No buttons, just motion. */
            event.type = XINE_EVENT_INPUT_MOUSE_MOVE;
            event.stream      = vwin->gui->stream;
            event.data        = &input;
            event.data_length = sizeof (input);
            xitk_gettime_tv (&event.tv);
            xine_event_send (vwin->gui->stream, &event);
          }
        }
      }
      return 1;
    case XITK_EV_DEL_WIN:
      gui_exit (NULL, vwin->gui);
      return 1;
    case XITK_EV_EXPOSE:
      if ((e->more == 0) && vwin->gui->vo_port) {
        XExposeEvent ev = { .type = Expose, .window = e->id, .count = 0, .x = e->x, .y = e->y, .width = e->w, .height = e->h };
        xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_EXPOSE_EVENT, (void *)&ev);
      }
      return 1;
    case XITK_EV_NEW_WIN:
    case XITK_EV_POS_SIZE:
      {
        int    h, w;

        pthread_mutex_lock (&vwin->mutex);

        h = vwin->output_height;
        w = vwin->output_width;

        vwin->output_width  = e->w;
        vwin->output_height = e->h;
        vwin->win_r.x = e->x;
        vwin->win_r.y = e->y;

        /* Keep geometry memory of windowed mode in sync. */
        if (vwin->mode & WINDOWED_MODE) {
          vwin->old_win_r.width  = vwin->win_r.width  = vwin->output_width;
          vwin->old_win_r.height = vwin->win_r.height = vwin->output_height;
        }

        /* reposition osd's now when the window actually has resized.
         * checking for vwin->output_{width,height} change alone is unsafe because
         * these fields may have set early by _vwin_adapt_size ().
         * anyway, try to skip this when window has moved only. */
        if ((h != vwin->output_height) || (w != vwin->output_width) || vwin->output_changed) {
          osd_update (vwin->gui);
          vwin->output_changed = 0;
        }

        oxine_adapt (vwin->gui->oxine);

        pthread_mutex_unlock (&vwin->mutex);
      }
      return 1;
    case XITK_EV_KEY_UP:
    case XITK_EV_KEY_DOWN:
      /* gui_handle_be_event () now does the menu. */
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        static const uint8_t tab_keys[XITK_KEY_LASTCODE] = {
          [XITK_KEY_UP] = OXINE_KEY_UP,
          [XITK_KEY_DOWN] = OXINE_KEY_DOWN,
          [XITK_KEY_LEFT] = OXINE_KEY_LEFT,
          [XITK_KEY_RIGHT] = OXINE_KEY_RIGHT,
          [XITK_KEY_PREV] = OXINE_KEY_PRIOR,
          [XITK_KEY_NEXT] = OXINE_KEY_NEXT,
          [XITK_KEY_RETURN] = OXINE_KEY_SELECT,
          [XITK_KEY_ESCAPE] = OXINE_KEY_ESCAPE
        };
        oe.type = OXINE_EVENT_KEY;
	oe.key = tab_keys[(uint8_t)e->utf8[1]] | ((e->type == XITK_EV_KEY_DOWN) ? 0 : OXINE_KEY_RELEASE);
        if ((oe.key & ~OXINE_KEY_RELEASE) &&
          !(e->qual & (MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD3 | MODIFIER_MOD4 | MODIFIER_MOD5)) &&
          _vwin_oxine_event (vwin, &oe))
          return 1;
        if ((e->utf8[1] == XITK_KEY_MENU) && vwin->separate_display)
          return 1;
      }
      /* fall through */
    default: ;
  }
  return gui_handle_be_event (vwin->gui, e);
}

static xitk_window_t *_vwin_new (xui_vwin_t *vwin, xitk_rect_t *r, int really_new) {
  uint32_t xinerama_flags =
#ifdef HAVE_XINERAMA
    vwin->xinerama_flags
#else
    0
#endif
    ;
  xitk_tagitem_t tags[] = {
    {XITK_TAG_WIN_FLAGS,
      ((XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED  | XITK_WINF_DECORATED | XITK_WINF_TASKBAR |
        XITK_WINF_PAGER | XITK_WINF_FULLSCREEN | XITK_WINF_DND | XITK_WINF_GRAB_POINTER |
        XITK_WINF_FENCED_IN | XITK_WINF_OVERRIDE_REDIRECT | XITK_WINF_LOCK_OPACITY) << 16) |
      (((vwin->new_mode & (WINDOWED_MODE | BORDER_MODE)) == (WINDOWED_MODE | BORDER_MODE))
        ? XITK_WINF_DECORATED : 0 /* XITK_WINF_OVERRIDE_REDIRECT */) |
      (vwin->separate_display ? 0 : (XITK_WINF_TASKBAR | XITK_WINF_PAGER)) |
      (((xinerama_flags & 16) && (vwin->new_mode & FULLSCR_XI_MODE)) ? 0 : XITK_WINF_FENCED_IN) |
      XITK_WINF_LOCK_OPACITY | XITK_WINF_DND
    },
    {XITK_TAG_GRAVITY, XITK_GRAVITY_STATIC},
    {XITK_TAG_TITLE, (uintptr_t)vwin->window_title},
    {XITK_TAG_RES_CLASS, (uintptr_t)"xine"},
    {XITK_TAG_RES_NAME,
      (uintptr_t)(!(vwin->new_mode & WINDOWED_MODE) ? vwin->res_name.fullscreen
                : !(vwin->new_mode & BORDER_MODE) ? vwin->res_name.borderless
                : vwin->res_name.normal)},
    {XITK_TAG_X, r->x},
    {XITK_TAG_Y, r->y},
    {XITK_TAG_WIDTH, r->width},
    {XITK_TAG_HEIGHT, r->height},
    {XITK_TAG_END, 0}
  };
  xitk_be_window_t *bewin;
  xitk_window_t *xwin;

  if (!really_new) {
    if ((bewin = vwin->win.bewin)) {
      if (vwin->gui->verbosity >= 2)
        printf ("video_window.resize (%d, %d, %d, %d).\n", r->x, r->y, r->width, r->height);
      bewin->set_props (bewin, tags + 5);
      return vwin->win.xwin;
    }
  }

  if (vwin->gui->verbosity >= 2)
    printf ("video_window.%s (%d, %d, %d, %d).\n", (vwin->win.xwin ? "replace" : "new"), r->x, r->y, r->width, r->height);
  bewin = vwin->video_be_display->window_new (vwin->video_be_display, tags);
  if (!bewin)
    return NULL;
  xwin = xitk_window_wrap_be_window (vwin->xitk, bewin);
  if (!xwin) {
    bewin->_delete (&bewin);
    return NULL;
  }
  vwin->oldwin = vwin->win;
  vwin->win.bewin = bewin;
  vwin->win.xwin = xwin;
  vwin->win.window = bewin->id;

  if (!vwin->separate_display) {
    /* avoid destroy notify from old window (triggers exit) */
    xitk_unregister_event_handler (vwin->xitk, &vwin->widget_key);
    vwin->widget_key = xitk_be_register_event_handler ("video_window", vwin->win.xwin,
      _vwin_handle_be_event, vwin, NULL, NULL);
    /* NOTE: XITK_TAG_RES_CLASS makes kwin use a desktop file named <res_class>.desktop to set the icon.
     * any subsequent attempt to set an icon has no effect then.
     * setting an icon _before_ leads to random fallback to default x icon. */
    xitk_window_set_window_icon (vwin->win.xwin, vwin->gui->icon);
  }
  if (vwin->gui->cursor_grabbed)
    xitk_window_flags (vwin->win.xwin, XITK_WINF_GRAB_POINTER, ~0u);

  if (vwin->gui->vo_port)
    xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_DRAWABLE_CHANGED, (void*)vwin->win.window);

  xitk_window_destroy_window (vwin->oldwin.xwin);

  xitk_window_set_role (vwin->win.xwin, vwin->gui->use_root_window ? XITK_WR_ROOT : XITK_WR_MAIN);

  vwin->oldwin.xwin = NULL;
  vwin->oldwin.bewin = NULL;
  vwin->oldwin.window = 0;

  return xwin;
}

/*
 * will modify/create video output window based on
 *
 * - fullscreen flags
 * - win_width/win_height
 *
 * will set
 * output_width/output_height
 * visible_width/visible_height/visible_aspect
 */
static void _vwin_adapt_size (xui_vwin_t *vwin, int really_new) {
  xitk_rect_t r = {0, 0, 0, 0};
  uint32_t mode_change;

  if (vwin->wid && !vwin->win.xwin) {
    if ((vwin->win.xwin = xitk_x11_wrap_window (vwin->xitk, vwin->video_be_display, vwin->wid))) {
      vwin->win.window = vwin->wid;
/*    vwin->win.bewin = vwin->win.xwin->bewin; */
    }
  }

  if (vwin->gui->use_root_window) { /* Using root window, but not really */

    vwin->win_r.x = vwin->win_r.y = 0;
    vwin->visible_width  = vwin->output_width  = vwin->fullscreen_width;
    vwin->visible_height = vwin->output_height = vwin->fullscreen_height;
    vwin->visible_aspect  = vwin->pixel_aspect = 1.0;

    if (!vwin->win.xwin) {
      xitk_rect_t r = {0, 0, vwin->fullscreen_width, vwin->fullscreen_height};

      vwin->mode &= ~(WINDOWED_MODE | FULLSCR_XI_MODE);
      vwin->mode |= FULLSCR_MODE;

      if (!vwin->wid) {
        _vwin_new (vwin, &r, really_new);
      } else {
        if (vwin->gui->verbosity >= 2)
          printf ("video_window.root.resize (%d, %d, %d, %d).\n", r.x, r.y, r.width, r.height);
        xitk_window_move_resize (vwin->win.xwin, &r);
        xitk_window_flags (vwin->win.xwin, XITK_WINF_LOCK_OPACITY, XITK_WINF_LOCK_OPACITY);
      }

      if (vwin->win.bewin)
        vwin->win.bewin->action (vwin->win.bewin, XITK_WA_CLEAR);
      xitk_window_flags (vwin->win.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
      if (vwin->win.bewin)
        vwin->win.bewin->action (vwin->win.bewin, XITK_WA_LOWER);

      if (vwin->wid && vwin->gui->vo_port)
        xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_DRAWABLE_CHANGED, (void*)vwin->win.window);
    }
    return;
  }

#ifdef HAVE_XF86VIDMODE
  if (vwin->XF86VidMode)
    _adjust_modeline(vwin);
#endif
#ifdef HAVE_XINERAMA
  _detect_xinerama_pos_size (vwin, &r);
#endif

  mode_change = vwin->mode ^ vwin->new_mode;

  vwin->visible_width  = vwin->fullscreen_width;
  vwin->visible_height = vwin->fullscreen_height;
  vwin->visible_aspect = vwin->pixel_aspect;

  /* Retrieve size/aspect from tvout backend, if it should be set */
  if (vwin->gui->tvout) {
    tvout_get_size_and_aspect (vwin->gui->tvout,
      &vwin->visible_width, &vwin->visible_height, &vwin->visible_aspect);
    tvout_get_size_and_aspect (vwin->gui->tvout,
      &r.width, &r.height, NULL);
    tvout_set_fullscreen_mode (vwin->gui->tvout,
      !(vwin->new_mode & WINDOWED_MODE) ? 1 : 0, vwin->visible_width, vwin->visible_height);
  }

#ifdef HAVE_XINERAMA
  /* ask for xinerama fullscreen mode */
  if ((vwin->xinerama_flags & 16) && (vwin->new_mode & FULLSCR_XI_MODE)) {

    vwin->mode = vwin->new_mode;
    /* open fullscreen window. */
    vwin->output_width    = r.width;
    vwin->output_height   = r.height;

  } else

#endif /* HAVE_XINERAMA */

  if (!(vwin->new_mode & WINDOWED_MODE)) {

    if (vwin->win.xwin)
      xitk_window_get_window_position (vwin->win.xwin, &vwin->old_win_r);
    vwin->mode = vwin->new_mode;
    /* open fullscreen window. */
#ifndef HAVE_XINERAMA
    r.width  = vwin->visible_width;
    r.height = vwin->visible_height;
#endif
    vwin->output_width    = r.width;
    vwin->output_height   = r.height;

  } else { /* WINDOWED_MODE */

#ifndef HAVE_XINERAMA
    r.width  = vwin->win_r.width;
    r.height = vwin->win_r.height;
#endif
    /* user sets window geom, move back to original location. */
    if (vwin->stream_resize_window == 0) {
      r.x = vwin->old_win_r.x;
      r.y = vwin->old_win_r.y;
    }
    vwin->old_win_r.width  = r.width;
    vwin->old_win_r.height = r.height;
    vwin->output_width  = r.width;
    vwin->output_height = r.height;

  }

  /* still same mode -> plain resize. */
  if (!(mode_change & (WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE)) && vwin->win.xwin) {
    /* resizing the video window may be necessary if the modeline or tv mode has just been switched. */ 
    if (vwin->new_mode & WINDOWED_MODE) {
      r.x = r.y = XITK_INT_KEEP;
      r.width = vwin->win_r.width;
      r.height = vwin->win_r.height;
    } else {
      r.width = vwin->visible_width;
      r.height = vwin->visible_height;
    }
    xitk_window_move_resize (vwin->win.xwin, &r);
    /* the above may have clipped values immediately. */
    xitk_window_get_window_position (vwin->win.xwin, &r);
    vwin->output_width = r.width;
    vwin->output_height = r.height;
    vwin->output_changed = 1;
    if (!(vwin->new_mode & WINDOWED_MODE)) {
      vwin->visible_width = r.width;
      vwin->visible_height = r.height;
    }
    vwin->mode = vwin->new_mode;
    if (vwin->gui->verbosity >= 2)
      printf ("video_window.%s.resize (%d, %d, %d, %d).\n",
        ((vwin->mode & WINDOWED_MODE) ? "windowed" : (vwin->mode & FULLSCR_MODE) ? "fullscreen" : "xinerama"),
        r.x, r.y, r.width, r.height);
    return;
  }

  if (!vwin->wid)
    _vwin_new (vwin, &r, really_new);
  vwin->mode = vwin->new_mode;

  {
    uint32_t flags = XITK_WINF_VISIBLE;
    if (vwin->hide_on_start == 1) {
      vwin->hide_on_start = -1;
      vwin->show = 0;
      flags = XITK_WINF_ICONIFIED;
    }
    xitk_window_raise_window (vwin->win.xwin);
    /* Map window. */
    pthread_mutex_unlock (&vwin->mutex);
    /* NOTE: this calls xitk_update_tree (), which tries to grab xitk lock :-/ */
    xitk_window_flags (vwin->win.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, flags);
    pthread_mutex_lock (&vwin->mutex);
    if (flags == XITK_WINF_VISIBLE) {
      _vwin_layer_above (vwin);
      /* inform the window manager that we are fullscreen. This info musn't be set for xinerama-fullscreen,
      otherwise there are 2 different window size for one fullscreen mode ! (kwin doesn't accept this) */
      if (!(vwin->mode & (WINDOWED_MODE | FULLSCR_XI_MODE)))
        xitk_window_flags (vwin->win.xwin, XITK_WINF_FULLSCREEN, XITK_WINF_FULLSCREEN);
    }
  }

  if (!(vwin->mode & WINDOWED_MODE)) {
    xitk_rect_t wr = {r.x, r.y, XITK_INT_KEEP, XITK_INT_KEEP};
    /* Waiting for visibility, avoid X error on some cases */
    /* _vwin_set_input_focus(vwin); */
    if (vwin->gui->verbosity >= 2)
      printf ("video_window.fullscreen.2.resize (%d, %d, %d, %d).\n", r.x, r.y, r.width, r.height);
    xitk_window_move_resize (vwin->win.xwin, &wr);
  }

  xitk_window_get_window_position (vwin->win.xwin, &vwin->win_r);

  oxine_adapt (vwin->gui->oxine);
}

static int get_default_mag (xui_vwin_t *vwin, int video_width, int video_height) {
  return (vwin->zoom_small_stream && video_width < 300 && video_height < 300) ? 2 : 1;
}

/*
 *
 */
static void _vwin_dest_size_cb (void *data,
  int video_width, int video_height, double video_pixel_aspect,
  int *dest_width, int *dest_height, double *dest_pixel_aspect)  {
  xui_vwin_t *vwin = data;

  if (!vwin)
    return;
  pthread_mutex_lock (&vwin->mutex);

  vwin->frame_width = video_width;
  vwin->frame_height = video_height;

  /* correct size with video_pixel_aspect */
  if (video_pixel_aspect >= vwin->pixel_aspect)
    video_width  = video_width * video_pixel_aspect / vwin->pixel_aspect + .5;
  else
    video_height = video_height * vwin->pixel_aspect / video_pixel_aspect + .5;

  if (vwin->stream_resize_window && (vwin->mode & WINDOWED_MODE)) {

    if (vwin->video_width != video_width || vwin->video_height != video_height) {

      if ((video_width > 0) && (video_height > 0)) {
        vwin_mag_t mag = get_default_mag (vwin, video_width, video_height);

        /* FIXME: this is supposed to give the same results as if a
         * video_window_set_mag(xmag, ymag) was called. Since _vwin_adapt_size()
         * check several other details (like border, xinerama, etc) this
         * may produce wrong values in some cases. (?)
         */
        *dest_width  = video_width * mag;
        *dest_height = video_height * mag;
        *dest_pixel_aspect = vwin->pixel_aspect;
        pthread_mutex_unlock (&vwin->mutex);
        return;
      }
    }
  }

  if (!(vwin->mode & WINDOWED_MODE)) {
    *dest_width  = vwin->visible_width;
    *dest_height = vwin->visible_height;
    *dest_pixel_aspect = vwin->visible_aspect;
  } else {
    *dest_width  = vwin->output_width;
    *dest_height = vwin->output_height;
    *dest_pixel_aspect = vwin->pixel_aspect;
  }

  pthread_mutex_unlock (&vwin->mutex);
}

/*
 * Set/Get magnification.
 */
static int _vwin_check_mag (xui_vwin_t *vwin) {
  if ((!(vwin->mode & WINDOWED_MODE))
/*
 * Currently, no support for magnification in fullscreen mode.
 * Commented out in order to not mess up current mag for windowed mode.
 *
#ifdef HAVE_XF86VIDMODE
     && !(vwin->XF86_modelines_count > 1)
#endif
 */
    )
    return 0;

  /* Allow mag only if video win is visible, so don't do something we can't see. */
  return !!(xitk_window_flags (vwin->win.xwin, 0, 0) & XITK_WINF_VISIBLE);
}

static void _vwin_frame_output_cb (void *data,
  int video_width, int video_height, double video_pixel_aspect,
  int *dest_x, int *dest_y, int *dest_width, int *dest_height, double *dest_pixel_aspect,
  int *win_x, int *win_y) {
  xui_vwin_t *vwin = data;
  int lock;
  xitk_t *xitk;

  if (!vwin)
    return;
  /* we are called from libxine video out thread without xitk lock
   * in way most cases, there will be not much to do here, and a
   * fast vwin only lock will be wnough. */
  pthread_mutex_lock (&vwin->mutex);
  xitk = vwin->gui->xitk;

  vwin->frame_width = video_width;
  vwin->frame_height = video_height;

  /* correct size with video_pixel_aspect */
  if (video_pixel_aspect >= vwin->pixel_aspect)
    video_width  = video_width * video_pixel_aspect / vwin->pixel_aspect + .5;
  else
    video_height = video_height * vwin->pixel_aspect / video_pixel_aspect + .5;

  /* Please do NOT remove, support will be added soon! */
#if 0
  double jitter;
  vwin->video_duration = video_duration;
  vwin->video_average  = 0.5 * vwin->video_average + 0.5 video_duration;
  jitter = ABS (video_duration - vwin->video_average) / vwin->video_average;
  if (jitter > EST_MAX_JITTER) {
    if (vwin->duration_valid > -EST_KEEP_VALID)
      vwin->duration_valid--;
  } else {
    if (vwin->duration_valid < EST_KEEP_VALID)
      vwin->duration_valid++;
    if (ABS (video_duration - vwin->use_duration) / video_duration > EST_MAX_DIFF)
      vwin->use_duration = video_duration;
  }
#endif

  lock = 0;
  if ((vwin->video_width ^ video_width) | (vwin->video_height ^ video_height)) {
    /* well. window resize will need some high level xitk stuff.
     * grabbing xitk lock _after_ vwin lock may freeze. so risk some
     * rare multiple resizes rather than hidden memory trashing with
     * failed resize or even a delayed crash. */
    pthread_mutex_unlock (&vwin->mutex);
    if (!xitk_lock (xitk, 152)) {
      pthread_mutex_lock (&vwin->mutex);
      lock = 1;
      if (vwin->gui->verbosity >= 2)
        printf ("video_window.frame_output.resize.relock ().\n");

      vwin->video_width  = video_width;
      vwin->video_height = video_height;

      if (vwin->stream_resize_window && (video_width > 0) && (video_height > 0)) {
        vwin_mag_t mag = get_default_mag (vwin, video_width, video_height);
        /* Prepare window size */
        vwin->win_r.width  = vwin->video_width * mag;
        vwin->win_r.height = vwin->video_height * mag;
        /* If actually ready to adapt window size, do it now */
        if (_vwin_check_mag (vwin))
          _vwin_adapt_size (vwin, 0);
      }
      /* oxine_adapt (vwin->gui->oxine); already part of _vwin_adapt_size (). */
    } else {
      /* xitk lock unavailable quickly, eg an event handler stuck inside xine_play ()
       * waiting for first frame, which is waiting for us here. defer resize to next
       * frame as last resort. */
      pthread_mutex_lock (&vwin->mutex);
      if (vwin->gui->verbosity >= 2)
        printf ("video_window.frame_output.resize.relock.failed.\n");
    }
  }

  *dest_x = 0;
  *dest_y = 0;

  if (!(vwin->mode & WINDOWED_MODE)) {
    *dest_width  = vwin->visible_width;
    *dest_height = vwin->visible_height;
    *dest_pixel_aspect = vwin->visible_aspect;
    /* TODO: check video size/fps/ar if tv mode and call _vwin_adapt_size if necessary */
  } else {
    *dest_width  = vwin->output_width;
    *dest_height = vwin->output_height;
    *dest_pixel_aspect = vwin->pixel_aspect;
  }

  *win_x = (vwin->win_r.x < 0) ? 0 : vwin->win_r.x;
  *win_y = (vwin->win_r.y < 0) ? 0 : vwin->win_r.y;

  pthread_mutex_unlock (&vwin->mutex);
  if (lock) {
    if (vwin->gui->verbosity >= 2)
      printf ("video_window.frame_output.resize.unlock ().\n");
    xitk_lock (xitk, 0);
  }
}

/*
 * very small X event loop for the second display
 */
static void *_vwin_second_display_loop (void *data) {
  xui_vwin_t *vwin = data;
  xitk_be_display_t *d;

  pthread_mutex_lock (&vwin->mutex);
  d = vwin->video_be_display;

  while (vwin->second_display_running) {
    xitk_window_t *xwin = vwin->win.xwin;
    xitk_be_event_t event;

    pthread_mutex_unlock (&vwin->mutex);
    if (d->next_event (d, &event, NULL, XITK_EV_ANY, 33)) {
      if (event.window && (xwin == event.window->data)) {
        xitk_lock (vwin->xitk, 1);
        _vwin_handle_be_event (vwin, &event);
        xitk_lock (vwin->xitk, 0);
      }
    }
    pthread_mutex_lock (&vwin->mutex);
  }
  pthread_mutex_unlock (&vwin->mutex);

  return NULL;
}

/** public ******************************************************************/

int video_window_is_separate_display (xui_vwin_t *vwin) {
  return vwin ? vwin->separate_display : 0;
}

void video_window_set_transient_for (xui_vwin_t *vwin, xitk_window_t *xwin) {
  if (!vwin || !xwin)
    return;
  if (vwin->gui->use_root_window || vwin->separate_display)
    return;
  pthread_mutex_lock (&vwin->mutex);
  xitk_window_set_transient_for_win(xwin, vwin->win.xwin);
  pthread_mutex_unlock (&vwin->mutex);
}

/*
void video_window_set_input_focus (xui_vwin_t *vwin) {
  if (!vwin)
    return;
  pthread_mutex_lock (&vwin->mutex);
  _vwin_set_input_focus (vwin);
  pthread_mutex_unlock (&vwin->mutex);
}
*/

int video_window_is_visible (xui_vwin_t *vwin) {
  if (!vwin)
    return 0;
  /* user may have changed this via task bar. */
  pthread_mutex_lock (&vwin->mutex);
  _vwin_is_visible (vwin);
  pthread_mutex_unlock (&vwin->mutex);
  return vwin->show;
}

void video_window_grab_input_focus(xui_vwin_t *vwin) {
  if (!vwin)
    return;

  pthread_mutex_lock (&vwin->mutex);
  if (vwin->gui->cursor_grabbed)
    xitk_window_flags (vwin->win.xwin, XITK_WINF_GRAB_POINTER, ~0u);
  if (_vwin_is_visible (vwin) > 1) {
    /* Give focus to video output window */
    xitk_window_set_input_focus (vwin->win.xwin);
  }
  pthread_mutex_unlock (&vwin->mutex);
}

void video_window_grab_pointer (xui_vwin_t *vwin, int grab) {
  if (!vwin)
    return;
  pthread_mutex_lock (&vwin->mutex);
  xitk_window_flags (vwin->win.xwin, XITK_WINF_GRAB_POINTER, (grab ? ~0u : 0));
  pthread_mutex_unlock (&vwin->mutex);
}

/*
 * Let the video driver override the selected visual
 */
void video_window_select_visual (xui_vwin_t *vwin) {
  XVisualInfo *vinfo = (XVisualInfo *) -1;

  /* this is done before creating initial video window */
  if (!vwin)
    return;
  if (vwin->gui->vo_port && !vwin->separate_display) {

    xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_SELECT_VISUAL, &vinfo);

    if (vinfo != (XVisualInfo *) -1) {
      if (! vinfo) {
        fprintf (stderr, _("videowin: output driver cannot select a working visual\n"));
        exit (1);
      }
      if (vwin->visual != vinfo->visual) {
        printf (_("videowin: output driver overrides selected visual to visual id 0x%lx\n"), vinfo->visual->visualid);
        xitk_x11_select_visual (vwin->xitk, vinfo->visual);
        vwin->visual = vinfo->visual;
        vwin->depth = vinfo->depth;
      }
    }
  }
}

void video_window_set_cursor_visibility (xui_vwin_t *vwin, int show_cursor) {
  if (!vwin)
    return;
  if (show_cursor) {
    _vwin_cursor_on (vwin);
  } else {
    if (vwin->gui->use_root_window)
      return;
    vwin->cursor_visible = 0;
    xitk_window_define_window_cursor (vwin->win.xwin, xitk_cursor_invisible);
  }
}

void video_window_set_cursor (xui_vwin_t *vwin, int cursor) {
  if (vwin && cursor) {
    vwin->current_cursor = cursor;
    if (vwin->cursor_visible) {
      vwin->gui->cursor_visible = 5;
      xitk_window_define_window_cursor (vwin->win.xwin,
        (vwin->current_cursor == 0) ? xitk_cursor_invisible :
        (vwin->current_cursor == CURSOR_ARROW) ? xitk_cursor_default :
        xitk_cursor_hand2);
    }
  }
}

void *video_window_get_xine_visual (xui_vwin_t *vwin, int *visual_type) {
  if (!vwin)
    return NULL;
  if (!vwin->video_be_display || !vwin->win.bewin)
    return NULL;

  switch (vwin->win.bewin->type) {
#if defined(HAVE_WAYLAND) && defined(XINE_VISUAL_TYPE_WAYLAND)
    case XITK_BE_TYPE_WAYLAND:
      *visual_type = XINE_VISUAL_TYPE_WAYLAND;
      vwin->xine_visual.wl.display           = (void *)vwin->video_be_display->id;
      vwin->xine_visual.wl.surface           = (void *)vwin->win.window;
      vwin->xine_visual.wl.frame_output_cb   = _vwin_frame_output_cb;
      vwin->xine_visual.wl.user_data         = vwin;
      break;
#endif
#if defined(HAVE_X11)
    case XITK_BE_TYPE_X11:
      {
        /* NOTE: bewin.XITK_TAG_SCREEN is valid after receiving XITK_EV_NEW_WIN only. */
        xitk_rect_t r = { .x = vwin->win_r.x, .y = vwin->win_r.y, .width = vwin->output_width, .height = vwin->output_height };
        vwin->xine_visual.x11.screen          = vwin->video_be_display->find_screen (vwin->video_be_display, &r);
      }
      *visual_type = XINE_VISUAL_TYPE_X11;
      vwin->xine_visual.x11.display           = (void *)vwin->video_be_display->id;
      vwin->xine_visual.x11.d                 = vwin->win.window;
      vwin->xine_visual.x11.dest_size_cb      = _vwin_dest_size_cb;
      vwin->xine_visual.x11.frame_output_cb   = _vwin_frame_output_cb;
      vwin->xine_visual.x11.user_data         = vwin;
      break;
#endif
    default:
      *visual_type = XINE_VISUAL_TYPE_NONE;
      return NULL;
  }
  return &vwin->xine_visual;
}

/*
 *
 */
uint32_t video_window_mode (xui_vwin_t *vwin, uint32_t mode_flags) {
  uint32_t d, r;

  if (!vwin)
    return 0;
  mode_flags &= WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE | BORDER_MODE | TOGGLE_MODE;

  pthread_mutex_lock (&vwin->mutex);
  /* user may have changed these through the window manager (alt-f3). */
  r = xitk_window_flags (vwin->win.xwin, 0, 0);
  vwin->mode &= ~FULLSCR_MODE;
  vwin->mode |= xitk_bitmove (r, XITK_WINF_FULLSCREEN, FULLSCR_MODE);
  if (!vwin->gui->use_root_window && (vwin->mode & WINDOWED_MODE)) {
    vwin->mode &= ~BORDER_MODE;
    vwin->mode |= xitk_bitmove (r, XITK_WINF_DECORATED, BORDER_MODE);
  }
  /* the new mode. */
  vwin->new_mode = (mode_flags & TOGGLE_MODE) ? vwin->mode ^ (mode_flags & ~TOGGLE_MODE) : mode_flags;
  /* block unsupported xinerama. */
#ifdef HAVE_XINERAMA
  if (!(vwin->xinerama_flags & 16))
#endif
    vwin->new_mode &= ~FULLSCR_XI_MODE;
  /* root window is always fullscreen. */
  if (vwin->gui->use_root_window)
    vwin->new_mode = (vwin->new_mode & ~(WINDOWED_MODE | FULLSCR_XI_MODE)) | FULLSCR_MODE;
  /* some exclusion logic. */
  r = ~vwin->mode & vwin->new_mode & (WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE);
  if (!r)
    r = vwin->new_mode & (WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE);
  if (!r)
    vwin->new_mode = (vwin->new_mode & ~(WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE)) | WINDOWED_MODE;
  else if (r & WINDOWED_MODE)
    vwin->new_mode &= ~(FULLSCR_MODE | FULLSCR_XI_MODE);
  else if (r & FULLSCR_MODE)
    vwin->new_mode &= ~(WINDOWED_MODE | FULLSCR_XI_MODE);
  else if (r & FULLSCR_XI_MODE)
    vwin->new_mode &= ~(WINDOWED_MODE | FULLSCR_MODE);
  /* what has changed? */
  d = vwin->mode ^ vwin->new_mode;
  vwin->watch_mode &= ~d;
  r = vwin->new_mode;

  if (d) {
    if (d & (WINDOWED_MODE | FULLSCR_MODE | FULLSCR_XI_MODE)) {
      _vwin_adapt_size (vwin, 1);
      _vwin_set_input_focus (vwin);
      osd_update (vwin->gui);
      /* Drawable has changed, update cursor visiblity */
      _vwin_cursor_on (vwin);
    }
    if ((d & BORDER_MODE) && (r & WINDOWED_MODE)) {
      xitk_window_flags (vwin->win.xwin, XITK_WINF_DECORATED, xitk_bitmove (r, BORDER_MODE, XITK_WINF_DECORATED));
      xitk_window_set_window_class (vwin->win.xwin,
        ((r & BORDER_MODE) ? vwin->res_name.normal : vwin->res_name.borderless), "xine");
      xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_DRAWABLE_CHANGED, (void*)vwin->win.window);
    }
  }
  pthread_mutex_unlock (&vwin->mutex);
  return r;
}

/*
 * hide/show video window
 */
void video_window_set_visibility (xui_vwin_t *vwin, int show_window) {
  if (!vwin)
    return;
  if (vwin->gui->use_root_window)
    return;

  if (vwin->gui->vo_port)
    xine_port_send_gui_data (vwin->gui->vo_port, XINE_GUI_SEND_VIDEOWIN_VISIBLE, (void *)(intptr_t)show_window);

  pthread_mutex_lock (&vwin->mutex);

  /* When shutting down (panel == NULL), unmap. Old kwin dislikes destroying iconified windows.
   * Otherwise, dont unmap both vwin and panel - user needs some handle to get back in. */
  vwin->show = show_window ? 2
             : !vwin->gui->panel ? 0
             : 1;

  /* Switching to visible: If new window size requested meanwhile, adapt window */
  if ((vwin->show > 1) && (vwin->mode & WINDOWED_MODE) &&
     (vwin->win_r.width != vwin->old_win_r.width || vwin->win_r.height != vwin->old_win_r.height))
    _vwin_adapt_size (vwin, 0);

  if (vwin->show > 1) {
    xitk_window_flags (vwin->win.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
    _vwin_layer_above (vwin);
    /* inform the window manager that we are fullscreen. This info musn't be set for xinerama-fullscreen,
       otherwise there are 2 different window size for one fullscreen mode ! (kwin doesn't accept this) */
    if (!(vwin->mode & (WINDOWED_MODE | FULLSCR_XI_MODE)))
      xitk_window_flags (vwin->win.xwin, XITK_WINF_FULLSCREEN, XITK_WINF_FULLSCREEN);
  } else {
    xitk_window_flags (vwin->win.xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, xitk_bitmove (vwin->show, 1, XITK_WINF_ICONIFIED));
  }

  /* User used '-H', now he want to show video window */
  if (vwin->hide_on_start == -1)
    vwin->hide_on_start = 0;

  pthread_mutex_unlock (&vwin->mutex);
}

xui_vwin_t *video_window_init (gGui_t *gui, int window_id,
  int borderless, const char *geometry, int hide_on_start,
  const char *prefered_visual, int use_x_lock_display, int install_colormap) {
  xui_vwin_t           *vwin;
  const char           *video_display_name;
  xitk_rect_t geom_r = { .x = 0, .y = 0, .width = -8192, .height = -8192 };

  if (!gui)
    return NULL;

  video_display_name = xine_config_register_string (gui->xine, "gui.video_display", "",
    _("Name of video display"),
    _("Use this setting to configure to which display xine will show videos. "
      "When left blank, the main display will be used. Setting this option to "
      "something like ':0.1' or ':1' makes possible to display video and control "
      "on different screens (very useful for TV presentations)."),
    CONFIG_LEVEL_ADV, NULL, CONFIG_NO_DATA);

  vwin = calloc (1, sizeof (*vwin));
  if (!vwin)
    return NULL;

  vwin->gui = gui;
  vwin->xitk = gui->xitk;
  {
    pthread_mutexattr_t attr;
    pthread_mutexattr_init (&attr);
    pthread_mutexattr_settype (&attr, PTHREAD_MUTEX_RECURSIVE);
    pthread_mutex_init (&vwin->mutex, &attr);
    pthread_mutexattr_destroy (&attr);
  }

  vwin->gui_be_display = xitk_get_gui_display (vwin->xitk);
  vwin->video_display = NULL;
  if (video_display_name && video_display_name[0]) {
    do {
      xitk_backend_t *video_backend;
      video_backend = xitk_backend_new (vwin->xitk, vwin->gui->verbosity);
      if (video_backend) {
        vwin->video_be_display = video_backend->open_display (video_backend,
          video_display_name, use_x_lock_display, 0, prefered_visual, install_colormap);
        video_backend->_delete (&video_backend);
        if (vwin->video_be_display) {
          vwin->video_display = (Display *)vwin->video_be_display->id;
          vwin->separate_display = 1;
          break;
        }
      }
      fprintf (stderr, _("Cannot open display '%s' for video. Falling back to primary display.\n"), video_display_name);
    } while (0);
  }
  if (vwin->video_display == NULL) {
    vwin->video_be_display = vwin->gui_be_display;
    vwin->video_display = (vwin->video_be_display->type == XITK_BE_TYPE_X11) ? (void *)vwin->video_be_display->id : NULL;
  }
  if (!vwin->video_display || !vwin->video_be_display) {
    fprintf (stderr, _("Cannot open any X11 display for video.\n"));
    free(vwin);
    return NULL;
  }
  {
    char *xrm_geometry = NULL;
    if (prefered_visual)
      vwin->prefered_visual = strdup(prefered_visual);
    xitk_x11_xrm_parse("xine",
                       geometry ? NULL : &xrm_geometry,
                       borderless ? NULL : &borderless,
                       vwin->prefered_visual ? NULL : &vwin->prefered_visual,
                       NULL);
    if (!gui->use_root_window) {
      if (!geometry)
        geometry = xrm_geometry;
      if (geometry && !xitk_x11_parse_geometry (geometry, &geom_r))
        printf(_("Bad geometry '%s'\n"), geometry);
    }
    free(xrm_geometry);
  }
  if (gui->use_root_window) {
    /*
     * Using root window mode don't allow
     * window geometry, so, reset those params.
     */
    geometry = NULL;
    borderless = 0;
  }

  vwin->win.window    = None;
  vwin->wid           = window_id;
  vwin->new_mode      =
  vwin->mode          = (vwin->gui->use_root_window) ? FULLSCR_MODE
                      : (borderless > 0) ? WINDOWED_MODE
                      : (WINDOWED_MODE | BORDER_MODE);
  vwin->show          = 2;
  vwin->widget_key    = 0;
  vwin->hide_on_start = hide_on_start;

  vwin->depth         = vwin->gui_be_display->get_depth (vwin->gui_be_display);
  vwin->visual        = (void *)vwin->gui_be_display->get_visual (vwin->gui_be_display);

  /* Currently, there no plugin loaded so far, but that might change */
  video_window_select_visual (vwin);
  if (vwin->separate_display) {
    vwin->video_be_display->lock (vwin->video_be_display);
    xitk_x11_find_visual (vwin->video_display, -1, vwin->prefered_visual, &vwin->visual, &vwin->depth);
    vwin->video_be_display->unlock (vwin->video_be_display);
  }
  vwin->fullscreen_width   = vwin->desktopWidth  = vwin->video_be_display->default_screen.r.width;
  vwin->fullscreen_height  = vwin->desktopHeight = vwin->video_be_display->default_screen.r.height;

  vwin->win_r.x = geom_r.x;
  vwin->win_r.y = geom_r.y;

  memcpy (vwin->window_title, "xine", 5);

#ifdef HAVE_XINERAMA
  _init_xinerama(vwin);
#endif

  vwin->visible_width  = vwin->fullscreen_width;
  vwin->visible_height = vwin->fullscreen_height;

  /* xclass hint for video window */
  vwin->res_name.normal     = _("xine Video Window");
  vwin->res_name.fullscreen = _("xine Video Fullscreen Window");
  vwin->res_name.borderless = _("xine Video Borderless Window");

  vwin->current_cursor = CURSOR_ARROW;

  vwin->stream_resize_window = xine_config_register_bool (vwin->gui->xine, "gui.stream_resize_window",
    1,
    _("New stream sizes resize output window"),
    CONFIG_NO_HELP, CONFIG_LEVEL_ADV, _vwin_resize_cb, vwin);

  vwin->zoom_small_stream = xine_config_register_bool (vwin->gui->xine, "gui.zoom_small_stream",
    0,
    _("Double size for small streams (require stream_resize_window)"),
    CONFIG_NO_HELP, CONFIG_LEVEL_ADV, _vwin_zoom_small_cb, vwin);

  if ((geom_r.width > 0) && (geom_r.height > 0)) {
    vwin->video_width  = geom_r.width;
    vwin->video_height = geom_r.height;
    /*
     * Force to keep window size.
     * I don't update the config file, i think this window geometry
     * user defined can be temporary.
     */
    vwin->stream_resize_window = 0;
  }
  else {
    vwin->video_width  = 768;
    vwin->video_height = 480;
  }
  vwin->old_win_r.width  = vwin->win_r.width  = vwin->video_width;
  vwin->old_win_r.height = vwin->win_r.height = vwin->video_height;

#ifdef HAVE_XF86VIDMODE
  if (xine_config_register_bool (vwin->gui->xine, "gui.use_xvidext",
    0,
    _("use XVidModeExtension when switching to fullscreen"),
    CONFIG_NO_HELP, CONFIG_LEVEL_EXP, CONFIG_NO_CB, CONFIG_NO_DATA)) {
    /*
     * without the "stream resizes window" behavior, the XVidMode support
     * won't work correctly, so we force it for each session the user wants
     * to have XVideMode on...
     *
     * FIXME: maybe display a warning message or so?!
     */
    vwin->stream_resize_window = 1;
    vwin->XF86VidMode = 1;
  }
#endif

  /*
   * Create initial video window with the geometry constructed above.
   */
  pthread_mutex_lock (&vwin->mutex);
  _vwin_adapt_size (vwin, 1);
  pthread_mutex_unlock (&vwin->mutex);

  /*
   * for plugins that aren't really bind to the window, it's necessary that the
   * vwin->win_r.x and vwin->win_r.y variables are set to *real* values, otherwise the
   * overlay will be displayed somewhere outside the window
   */
  if (vwin->win.xwin) {
    if (geometry && (geom_r.x > -8192) && (geom_r.y > -8192)) {
      vwin->win_r.x = vwin->old_win_r.x = geom_r.x;
      vwin->win_r.y = vwin->old_win_r.y = geom_r.y;
      {
        xitk_rect_t r = {vwin->win_r.x, vwin->win_r.y, vwin->video_width, vwin->video_height};
        if (vwin->gui->verbosity >= 2)
          printf ("video_window.init.resize (%d, %d, %d, %d).\n", r.x, r.y, r.width, r.height);
        xitk_window_move_resize (vwin->win.xwin, &r);
      }
    }
    else {
      xitk_window_get_window_position (vwin->win.xwin, &vwin->win_r);
    }
  }

  if (vwin->separate_display) {
    vwin->second_display_running = 1;
    pthread_create (&vwin->second_display_thread, NULL, _vwin_second_display_loop, vwin);
  }

  vwin->pixel_aspect = vwin->video_be_display->default_screen.ratio;
#ifdef DEBUG
    printf("pixel_aspect: %f\n", vwin->pixel_aspect);
#endif

  return vwin;
}

/*
 * Necessary cleanup
 */
void video_window_exit (xui_vwin_t *vwin) {
  if (!vwin)
    return;

#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (vwin->gui->xine, NULL, NULL, vwin, sizeof (*vwin));
#endif

  pthread_mutex_lock (&vwin->mutex);
  vwin->second_display_running = 0;
#ifdef HAVE_XF86VIDMODE
  /* Restore original VidMode */
  if (vwin->XF86VidMode)
    xitk_change_video_mode(vwin->xitk, vwin->win.xwin, -1, -1);
#endif
  pthread_mutex_unlock (&vwin->mutex);

  if (vwin->win.bewin)
    vwin->win.bewin->action (vwin->win.bewin, XITK_WA_CLEAR);

  if (vwin->separate_display) {
    pthread_join (vwin->second_display_thread, NULL);
  } else {
    xitk_unregister_event_handler (vwin->xitk, &vwin->widget_key);
  }
  xitk_window_destroy_window (vwin->win.xwin);
  vwin->win.window = 0;
  vwin->win.bewin = NULL;
  vwin->win.xwin = NULL;

  pthread_mutex_destroy (&vwin->mutex);

  if (vwin->separate_display && vwin->video_be_display)
    vwin->video_be_display->close (&vwin->video_be_display);

  free(vwin->prefered_visual);

  free (vwin);
}

vwin_mag_t video_window_set_mag (xui_vwin_t *vwin, vwin_mag_t mag) {
  if (!vwin)
    return 0;

  pthread_mutex_lock (&vwin->mutex);
  if (mag != -1 * VWIN_MAG_1) {
    int nx, ny;

    if (!_vwin_check_mag (vwin)) {
      pthread_mutex_unlock (&vwin->mutex);
      return 0;
    }
    if (mag < 0)
      mag = vwin_mag_mul (-mag,
        (vwin_mag_get (vwin->output_width, vwin->video_width) + vwin_mag_get (vwin->output_height, vwin->video_height)) >> 1);
    nx = vwin_mag_mul (vwin->video_width, mag);
    ny = vwin_mag_mul (vwin->video_height, mag);
    /* safe _vwin_calc_mag_win_size () */
    if (XITK_0_TO_MAX_MINUS_1 (nx - 50, 5000))
      vwin->win_r.width = nx;
    if (XITK_0_TO_MAX_MINUS_1 (ny - 50, 5000))
      vwin->win_r.height = ny;
    _vwin_adapt_size (vwin, 0);
  }
  mag = (vwin_mag_get (vwin->output_width, vwin->video_width) + vwin_mag_get (vwin->output_height, vwin->video_height)) >> 1;
  pthread_mutex_unlock (&vwin->mutex);
  return mag;
}

long int video_window_reset_ssaver (xui_vwin_t *vwin, int suspend) {

  long int idle = 0;

  if (!vwin)
    return 0;
  /* this is cleared temporarily while key binding editor window is open,
   * to not make the screen saver reset fake keys echoes trash the key recording.
   * but this will trash screen saver suspend/resume switches instead.
   * xitk shall now be smart enough to live without that hack.
  if (!vwin->gui->ssaver_enabled)
    return 0;
  */

  if (vwin->gui->flags & XUI_FLAG_save_screen)
    suspend = 0;
  if (suspend)
    suspend = vwin->gui->ssaver_timeout;
  if (vwin->video_be_display) {
    if (vwin->video_be_display->reset_screen_saver)
      idle = vwin->video_be_display->reset_screen_saver(vwin->video_be_display, suspend);
#if 0
  } else {
    idle = xitk_reset_screen_saver(vwin->xitk, suspend);
#endif
  }

  return idle;
}

void video_window_get_size (xui_vwin_t *vwin, int size[2], int what) {
  if (!size)
    return;
  if (!vwin)
    what = 0;

  switch (what) {
    case VWIN_SIZE_FRAME:
      /* fall back to meta info */
      size[0] = vwin->frame_width
              ? vwin->frame_width
              : (int)xine_get_stream_info (vwin->gui->stream, XINE_STREAM_INFO_VIDEO_WIDTH);
      size[1] = vwin->frame_height
              ? vwin->frame_height
              : (int)xine_get_stream_info (vwin->gui->stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
      break;
    case VWIN_SIZE_VISIBLE:
      size[0] = vwin->visible_width;
      size[1] = vwin->visible_height;
      break;
    case VWIN_SIZE_OUTPUT:
      size[0] = vwin->output_width;
      size[1] = vwin->output_height;
      break;
    default:
      size[0] = size[1] = 0;
  }
}

void video_window_set_mrl (xui_vwin_t *vwin, const char *mrl) {
  if (!vwin || !mrl)
    return;
  if (!mrl[0])
    return;
  if (!memcmp (vwin->window_title, "xine: ", 6) && !strcmp (vwin->window_title + 6, mrl))
    return;
  memcpy (vwin->window_title, "xine: ", 6);
  strlcpy (vwin->window_title + 6, mrl, sizeof (vwin->window_title) - 6);
  xitk_window_set_window_title (vwin->win.xwin, vwin->window_title);
}
