/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include <QApplication>
#include <QDBusConnection>
#include <QDBusError>
#include <QDebug>

#include "ukccsessionserver.h"
#include "session_adaptor.h"
#include "screenStruct.h"

#include "plugins/keyboard/keyboardinterface.h"
#include "plugins/mouse/mouseinterface.h"
#include "plugins/area/areainterface.h"
#include "plugins/default/defaultinterface.h"
#include "plugins/autoboot/autostartinterface.h"
#include "plugins/notice/noticeinterface.h"
#include "plugins/about/aboutinterface.h"
#include "plugins/datetime/datetimeinterface.h"
#include "plugins/wallpaper/wallpaperinterface.h"
#include "plugins/screenlock/screenlockinterface.h"
#include "plugins/shortcut/shortcutinterface.h"
#include "plugins/screensaver/screensaverinterface.h"
#include "plugins/vino/vinointerface.h"

void registerPlugins(const QString &path, QObject *object, QDBusConnection &connection) {
    if (!connection.registerObject(path, object, QDBusConnection::ExportAllSlots |
                                            QDBusConnection::ExportAllProperties |
                                            QDBusConnection::ExportAllSignals))
        qCritical() << QString("QDbus register object %1 failed reason:").arg(path) << connection.lastError();
}

int main(int argc, char *argv[])
{
    QApplication app(argc, argv);
    QCoreApplication::setOrganizationName("Kylin Team");
    QCoreApplication::setApplicationName("ukcc-session-service");
    qRegisterMetaType<ScreenConfig>("ScreenConfig");
    qDBusRegisterMetaType<ScreenConfig>();

    QTranslator translator;
    translator.load("/usr/share/ukui-control-center/shell/res/i18n/" + QLocale::system().name());
    app.installTranslator(&translator);

    ukccSessionServer service;
    new InterfaceAdaptor(&service);

    QDBusConnection sessionBus = QDBusConnection::sessionBus();
    if (!sessionBus.registerService("org.ukui.ukcc.session")) {
        qCritical() << "QDbus register service failed reason:" << sessionBus.lastError();
        exit(1);
    }

    if (!sessionBus.registerObject("/", &service)) {
        qCritical() << "QDbus register object failed reason:" << sessionBus.lastError();
        exit(2);
    }

    std::map <QString, QObject*> plugins;
    plugins["/Area"] = new AreaInterface;
    plugins["/Datetime"] = new DatetimeInterface;
    plugins["/KeyBoard"] = new KeyBoardInterface;
    plugins["/Mouse"] = new MouseInterface;
    plugins["/Default"] = new DefaultInterface;
    plugins["/Autoboot"] = new AutoStartInterface;
    plugins["/Notice"] = new NoticeInterface;
    plugins["/About"] = new AboutInterface;
    plugins["/Wallpaper"] = new WallpaperInterface;
    plugins["/Screenlock"] = new ScreenlockInterface;
    plugins["/Shortcut"] = new ShortcutInterface;
    plugins["/Screensaver"] = new ScreensaverInterface;
    plugins["/Vino"] = new VinoInterface;

    for (std::map<QString, QObject*>::value_type plugin : plugins)
        registerPlugins(plugin.first, plugin.second, sessionBus);

    return app.exec();
}
