/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
#include "first.h"

/* Test CURLINFO_SCHEME */

static CURLcode test_lib1536(const char *URL)
{
  CURL *curl, *dupe = NULL;
  char *scheme;
  CURLcode result = CURLE_OK;

  global_init(CURL_GLOBAL_ALL);

  easy_init(curl);

  /* Test that scheme is properly initialized on curl_easy_init.
   */

  result = curl_easy_getinfo(curl, CURLINFO_SCHEME, &scheme);
  if(result) {
    curl_mfprintf(stderr,
                  "%s:%d curl_easy_getinfo() failed with code %d (%s)\n",
                  __FILE__, __LINE__, result, curl_easy_strerror(result));
    goto test_cleanup;
  }
  if(scheme) {
    curl_mfprintf(stderr,
                  "%s:%d scheme init failed; expected NULL\n",
                  __FILE__, __LINE__);
    result = CURLE_FAILED_INIT;
    goto test_cleanup;
  }

  easy_setopt(curl, CURLOPT_URL, URL);

  result = curl_easy_perform(curl);
  if(result) {
    curl_mfprintf(stderr,
                  "%s:%d curl_easy_perform() failed with code %d (%s)\n",
                  __FILE__, __LINE__, result, curl_easy_strerror(result));
    goto test_cleanup;
  }

  /* Test that a scheme is properly set after receiving an HTTP resource.
   */

  result = curl_easy_getinfo(curl, CURLINFO_SCHEME, &scheme);
  if(result) {
    curl_mfprintf(stderr,
                  "%s:%d curl_easy_getinfo() failed with code %d (%s)\n",
                  __FILE__, __LINE__, result, curl_easy_strerror(result));
    goto test_cleanup;
  }
  if(!scheme || memcmp(scheme, "http", 5) != 0) {
    curl_mfprintf(stderr, "%s:%d scheme of http resource is incorrect; "
                  "expected 'http' but is %s\n",
                  __FILE__, __LINE__,
                  (scheme == NULL ? "NULL" : "invalid"));
    result = CURLE_HTTP_RETURNED_ERROR;
    goto test_cleanup;
  }

  /* Test that a scheme is properly initialized on curl_easy_duphandle.
   */

  dupe = curl_easy_duphandle(curl);
  if(!dupe) {
    curl_mfprintf(stderr, "%s:%d curl_easy_duphandle() failed\n",
                  __FILE__, __LINE__);
    result = CURLE_FAILED_INIT;
    goto test_cleanup;
  }

  result = curl_easy_getinfo(dupe, CURLINFO_SCHEME, &scheme);
  if(result) {
    curl_mfprintf(stderr,
                  "%s:%d curl_easy_getinfo() failed with code %d (%s)\n",
                  __FILE__, __LINE__, result, curl_easy_strerror(result));
    goto test_cleanup;
  }
  if(scheme) {
    curl_mfprintf(stderr, "%s:%d scheme init failed; expected NULL\n",
                  __FILE__, __LINE__);
    result = CURLE_FAILED_INIT;
    goto test_cleanup;
  }

  /* Test that a scheme is properly initialized on curl_easy_reset.
   */

  curl_easy_reset(curl);

  result = curl_easy_getinfo(curl, CURLINFO_SCHEME, &scheme);
  if(result) {
    curl_mfprintf(stderr,
                  "%s:%d curl_easy_getinfo() failed with code %d (%s)\n",
                  __FILE__, __LINE__, result, curl_easy_strerror(result));
    goto test_cleanup;
  }
  if(scheme) {
    curl_mfprintf(stderr, "%s:%d scheme init failed; expected NULL\n",
                  __FILE__, __LINE__);
    result = CURLE_FAILED_INIT;
    goto test_cleanup;
  }

test_cleanup:
  curl_easy_cleanup(curl);
  curl_easy_cleanup(dupe);
  curl_global_cleanup();
  return result;
}
